/*
 * Copyright (C) 2010  ed <tripflag@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License v2
 * (version 2) as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, refer to the following URL:
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt
 */
package tervis;

import java.awt.Color;
import java.awt.Font;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;

/**
 * Supplies a way to draw a graph using standard Swing controls.
 * @author ed <tripflag@gmail.com>
 */
public class Graph {
	private int delay;
	private final int SLEEP = 200;
	private final int MARGIN_X = 8;
	private final int MARGIN_Y = 24;
	private final double FONTSIZE = 1.5;
	private javax.swing.JComponent p;
	private javax.swing.JButton[] column;
	private javax.swing.JLabel[] hilighter;
	private Color colorDefault, colorHilight;
	private int comparisons, moves;
	private int[] oldHilight;
	private JLabel jStat;

	/**
	 * Creates a new Graph within the provided parent swingcontrol
	 * @param parent container control for the grapf
	 * @param columns how many columns to display
	 */
	public Graph(javax.swing.JPanel parent) {
		p = parent;
		delay = SLEEP;
		column = new JButton[0];
		comparisons = 0;
		moves = 0;

		// Statistics screens
		jStat = new JLabel("Hello");
		jStat.setSize(200, 32);
		jStat.setLocation(8, 8);
		jStat.setVerticalAlignment(JLabel.TOP);
		p.add(jStat);
	}

	/**
	 * (re)initialize the controls, forked in case
	 * the number of columns should change on runtime
	 * @param columns amount of columns to display
	 */
	private void prepare(int columns) {

		// Erase the old controls
		for (int a = 0; a < column.length; a++) {
			p.remove(hilighter[a]);
			p.remove(column[a]);
		}

		// Initizlae arrays
		column = new JButton[columns];
		hilighter = new JLabel[columns];

		// Prepare the actual controls
		for (int a = 0; a < columns; a++) {

			// Create the column itself
			column[a] = new JButton("F");
			column[a].setVerticalAlignment(JButton.BOTTOM);
			column[a].setHorizontalAlignment(JButton.CENTER);
			column[a].setBorder(BorderFactory.createRaisedBevelBorder());
			p.add(column[a]);

			// Create an underliner to emphasize visualization
			hilighter[a] = new JLabel();
			hilighter[a].setOpaque(true);
			hilighter[a].setVisible(false);
			hilighter[a].setBackground(hilighter[a].getForeground());
			p.add(hilighter[a]);

			// Setting foreground to background makes the background inverted;
			// this is better than setting an explicit colour as we don't know
			// the colour pallette of the user's environment.

		}
		colorDefault = column[0].getBackground();
		colorHilight = column[0].getForeground();
		oldHilight = new int[0];
	}

	/**
	 * Draws the provided values as a graph
	 * @param items set of values to display
	 */
	public void draw(int[] items) {

		// Validate column count
		if (items.length != column.length) {
			prepare(items.length);
		}

		// Calculate colum width by subtracting (columns+1 * margin) from span
		int baseWidth = (p.getWidth() - MARGIN_X * (column.length + 1)) / column.length;

		// Height is more trivial; subtract two vertical margins from span
		int baseHeight = (p.getHeight() - MARGIN_Y * 2);

		// Divide height by maxvalue to get the height unit
		int heightUnit = baseHeight / Math.max(1, getMaxVal(items));

		// Also, somewhat dynamically scaling font size
		double fontSize = baseWidth / FONTSIZE;
		Font font = new Font("sans-serif", 0, (int) fontSize);

		for (int a = 0; a < items.length; a++) {

			// Position this after the previous column + a margin
			int itemLeft = (baseWidth + MARGIN_X) * a + MARGIN_X;

			// Decide height using the column value and height unit
			int itemHeight = heightUnit * items[a];

			// Set the size, position and font
			column[a].setSize(baseWidth, itemHeight);
			column[a].setLocation(itemLeft, p.getHeight() - MARGIN_Y - itemHeight);
			column[a].setText(items[a] + "");
			column[a].setFont(font);

			// We might need a hilighter here as well
			hilighter[a].setSize(baseWidth, MARGIN_Y);
			hilighter[a].setLocation(itemLeft, p.getHeight() - MARGIN_Y);
		}
		sleep(delay);
	}

	/**
	 * Visually swap two values
	 * @param a column index A
	 * @param b column index B
	 */
	public void swap(int ca, int cb) {
		// Give the relative distance the most weight in stepsize
		// calculation, but need a lambda to cater for shorter moves
		java.awt.Rectangle ra = column[ca].getBounds();
		java.awt.Rectangle rb = column[cb].getBounds();
		int steps = (8 + Math.abs(ca-cb) * 2) / 2;
		int step = (rb.x - ra.x) / steps;

		for (int a = 0; a < steps; a++) {
			column[ca].setLocation(ra.x + a*step, ra.y);
			column[cb].setLocation(rb.x + a*-step, rb.y);
			sleep(delay/10);
		}

		// So we don't mess up the order in the array:
		column[ca].setBounds(ra.x, rb.y, ra.width, rb.height);
		column[cb].setBounds(rb.x, ra.y, rb.width, ra.height);
		String sa = column[ca].getText();
		String sb = column[cb].getText();
		column[ca].setText(sb);
		column[cb].setText(sa);
	}

	/**
	 * Returns the largest vaule in the provided array
	 * @param items set to iterate
	 * @return the largest value
	 */
	private int getMaxVal(int[] items) {
		int ret = 0;
		for (int val : items) {
			ret = Math.max(val, ret);
		}
		return Math.max(1, ret);
	}

	/**
	 * Hilight zero or more columns
	 * @param indices columns to hilight
	 */
	public void hilight(int... indices) {
		hilights(delay, indices);
	}

	/**
	 * Hilight zero or more columns
	 * @param delay msec wait after each hilight
	 * @param indices columns to hilight
	 */
	public void hilights(int delay, int... indices) {
		for (int indice : oldHilight) {
			// See comments below :3
			hilighter[indice].setVisible(false);
			column[indice].setEnabled(true);
			column[indice].setBackground(colorDefault);
		}
		for (int indice : indices) {

			// Show the hilighter (underline)
			hilighter[indice].setVisible(true);

			// Change colour palette
			column[indice].setEnabled(false);

			// Fill border with !background
			column[indice].setBackground(colorHilight);
		}
		oldHilight = indices;
		sleep(delay);
	}

	/**
	 * Fancy slide effect on sortan finish
	 */
	public void extremerice() {
		//for (int a = 0; a < column.length; a++) {
		//	hilights(33, a);
		//}
		//hilight();
		int[] all = new int[column.length];
		for (int a = 0; a < all.length; a++){
			all[a] = a;
		}
		hilight(all);
		hilight();
	}

	/**
	 * Sleep for i msec
	 * @param i duration
	 */
	private void sleep(int i) {
		try {
			Thread.sleep(i);
		} catch (Exception ex) {
		}
	}

	/**
	 * Reset the comparison and move counters
	 */
	public void reset() {
		comparisons = 0;
		moves = 0;
	}

	/**
	 * Increases (and displays) comparison and move count
	 * @param cmp comparisons counter increment
	 * @param mov moves counter increment
	 */
	public void increase(int cmp, int mov) {
		// Arguably bad structuring, but hear me out...
		// Thanks to the differing nature of various sorting
		// algorithms, this is actually the cleanest approach.
		// Prove me wrong /g/.
		comparisons += cmp;
		moves += mov;
		jStat.setText(
			"Comparisons / Moves:   "
			+ comparisons + " . "
			+ moves);
	}

	/**
	 * Reads the current rendering delay factor
	 * @return well what do you think
	 */
	public int getDelay() {
		return delay;
	}
	/**
	 * Changes the rendering delay factor
	 * @param i the new delay (ms)
	 */
	public void setDelay(int i) {
		delay = i;
	}
}
