﻿/*
 * Slidan - An eightpuzzle solver and visualizer in C#
 * Copyright (C) 2010  ed (tripflag@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License v2
 * (version 2) as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, refer to the following url:
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
namespace Slidan {
    
    /// <summary>
    /// The visual interfact
    /// </summary>
    class SDialogue {
        
        Panel pn; //panel containing tiles
        Form form; //application window
        Button input; //keyboard hook
        Button[] tile; //a single tile
        Point[] position; //position of tile n

        int[] map; //initial task
        char[] solution; //solution
        public SDialogue(int[] map) {

            // Create a new window
            form = new Form();
            form.StartPosition = FormStartPosition.CenterScreen;
            form.Size = new System.Drawing.Size(500, 400);
            form.Padding = new Padding(12, 8, 12, 8);
            form.Text = "IT'S OVER, JAVA IS FINISHED";

            // Create the input interface
            input = new Button();
            input.Location = new Point(-100, -100);
            input.KeyDown += new KeyEventHandler(input_KeyDown);
            form.Controls.Add(input);
            input.Visible = true;

            // Create the panel
            pn = new Panel();
            pn.Dock = DockStyle.Fill;
            pn.Visible = true;

            // Create each tile
            tile = new Button[9];
            position = new Point[9];
            for (int a = 0; a < tile.Length; a++) {
                tile[a] = new Button();
                tile[a].Visible = a > 0;
                tile[a].Text = a.ToString();
                tile[a].GotFocus += new EventHandler(SDialogue_GotFocus);
                pn.Controls.Add(tile[a]);
            }

            // Assign methods and controls
            pn.Resize += new EventHandler(pn_Resize);
            form.Controls.Add(pn);
            solution = new char[0];
            this.map = map;
            reset(map);
        }
        private void reset(int[] map) {
            if (map.Length == 0) {
                // Start from clean slate
                position[0] = new Point(2, 2);
                for (int a = 0; a < tile.Length - 1; a++)
                    position[a + 1] = new Point(a % 3, a / 3);
            } else {
                // Based on solved task
                for (int a = 0; a < map.Length; a++) {
                    position[map[a]] = new Point(a % 3, a / 3);
                }
            }
            // Redraw
            pn_Resize(null, null);
        }

        public void run() {
            // Show app, wait 'til closed
            form.ShowDialog();
        }
        public void run(string str) {
            // Store solution, then run()
            solution = str.ToCharArray();
            form.ShowDialog();
        }

        void SDialogue_GotFocus(object sender, EventArgs e) {
            // User clicked a tile, redirect focus to `input'
            input.Focus();
        }
        void input_KeyDown(object sender, KeyEventArgs e) {

            // This really doesn't need any comments
            if (e.KeyCode == Keys.W) move(MOVE_U);
            if (e.KeyCode == Keys.S) move(MOVE_D);
            if (e.KeyCode == Keys.A) move(MOVE_L);
            if (e.KeyCode == Keys.D) move(MOVE_R);
            if (e.KeyCode == Keys.P) play();
            if (e.KeyCode == Keys.L) reset(map);
            if (e.KeyCode == Keys.N) reset(new int[0]);
            if (e.KeyCode == Keys.T) {
                // Start new instance in `silent' mode, kill this one
                System.Diagnostics.Process.Start(Application.ProductName + ".exe", "silent");
                System.Diagnostics.Process.GetCurrentProcess().Kill();
            }
            if (e.KeyCode == Keys.F1) MessageBox.Show(
                "Use `WASD' to play manually\r\n" +
                "Press `F11' for fullscreen\r\n\r\n" +
                "Press `P' to simulate (Play)\r\n" +
                "Press `L' to reload (Load)\r\n" +
                "Press `N' to reset (New)\r\n" +
                "Press `T' to restart (Thread)");
            
            // Toggle kiosk modo
            if (e.KeyCode == Keys.F11) form.WindowState =
                form.WindowState == FormWindowState.Normal ?
                FormWindowState.Maximized :
                FormWindowState.Normal;

            pn_Resize(null, null);
        }
        void pn_Resize(object sender, EventArgs e) {

            //Calculate tile size, margins etc
            Size span = pn.Size;
            Point margin = new Point(3, 2);
            Size size = new Size(
                (span.Width - 2 * margin.X) / 3,
                (span.Height - 2 * margin.Y) / 3);

            // Resize each tile
            for (int a = 0; a < tile.Length; a++) {
                tile[a].Size = size;
                tile[a].Font = new Font("sans-serif", size.Width / 3);
                tile[a].Location = new Point(
                    position[a].X * (size.Width + margin.X),
                    position[a].Y * (size.Height + margin.Y));
            }
        }



        // I could define these in the class header,
        // but what's the point. It's not like anyone
        // will read this code anyway.
        const int MOVE_U = 0;
        const int MOVE_D = 1;
        const int MOVE_L = 2;
        const int MOVE_R = 3;
        void move(int dir) {

            // Move a single tile
            Point to = position[0];
            Point from = to;
            if (dir == MOVE_U) from.Y++;
            if (dir == MOVE_D) from.Y--;
            if (dir == MOVE_L) from.X++;
            if (dir == MOVE_R) from.X--;

            // Validate within bounds
            if (from.X < 0 || from.Y < 0 ||
                from.X > 2 || from.Y > 2)
                return;
            
            // Find tile to move
            int ofs = tileAt(from);
            
            // Swap blank and tile
            Point tmp = position[0];
            position[0] = position[ofs];
            position[ofs] = tmp;
            
            // BLING BLANG LOL
            fancyMove(ofs);

            // Redraw if fancyMove fails
            // (happens when user hammers WASD)
            pn_Resize(null, null);
        }
        int tileAt(Point pos) {
            // Search for tile with given position
            for (int a = 0; a < position.Length; a++)
                if (position[a].X == pos.X &&
                    position[a].Y == pos.Y)
                    return a;
            return -1;
        }
        void fancyMove(int i) {
            // Visually move tile
            double steps = 10;
            Point to = tile[0].Location;
            Point from = tile[i].Location;
            tile[0].Location = from;
            for (int a = 0; a < steps; a++) {
                tile[i].Location = new Point(
                    (int)Math.Round(from.X + ((to.X - from.X) / steps) * a),
                    (int)Math.Round(from.Y + ((to.Y - from.Y) / steps) * a));
                System.Threading.Thread.Sleep(10);
                Application.DoEvents();
            }
        }



        void play() {
            // Play solution
            if (solution.Length == 0) return;
            foreach (char sol in solution) {
                System.Threading.Thread.Sleep(150);
                if (sol == 'd') move(MOVE_U);
                if (sol == 'u') move(MOVE_D);
                if (sol == 'r') move(MOVE_L);
                if (sol == 'l') move(MOVE_R);
                // When I wrote the java implementation, WASD (arrow keys)
                // were inverted relative to this version. This bridges it.
                Application.DoEvents();
            }
        }
    }
}
