//
//  UJETObject.swift
//  ExampleApp
//
//  Copyright © UJET. All rights reserved.
//

import JWT
import UJETKit

import CoreLocation
import CoreBluetooth

public class UJETObject: NSObject, UJETDelegate {
    // MARK: Custom Data
    internal private(set) lazy var locationManager: CLLocationManager = {
        let lm = CLLocationManager()
        lm.delegate = self
        lm.desiredAccuracy = kCLLocationAccuracyNearestTenMeters
        lm.distanceFilter = CLLocationDistance(500)
        lm.requestWhenInUseAuthorization()
        lm.startUpdatingLocation()
        return lm
    }()
    
    internal private(set) lazy var bluetoothManager = CBCentralManager(delegate: self,
                                                                     queue: nil,
                                                                     options: [CBCentralManagerOptionShowPowerAlertKey: 0])
    var bluetoothState: String?
    var location: String?
    
    // MARK: UJETDelegate
    
    public func signPayload(_ payload: [AnyHashable : Any]?, payloadType: UjetPayloadType, success: @escaping (String?) -> Void, failure: @escaping (Error?) -> Void) {
        if payloadType == .authToken {
            signAuthToken(inLocal: payload, success: success, failure: failure)
            
        } else if payloadType == .customData {
            signCustomData(inLocal: payload, success: success, failure: failure)
        }
    }
    
    private func signAuthToken(inLocal payload: [AnyHashable : Any]?, success: @escaping (String?) -> Void, failure: @escaping (Error?) -> Void) {
        var payloadData = payload!
        
        let userData = UserDefaults.standard.object(forKey: "user-data") as? [AnyHashable : Any]
        for (k, v) in userData! { payloadData[k] = v }
        payloadData["iat"] = Date().timeIntervalSince1970 // required
        payloadData["exp"] = Double((Date().timeIntervalSince1970 + 600)) // required
        
        if let signedToken = encodeJWT(payloadData) {
            success(signedToken)
        } else {
            let userInfo = [NSLocalizedDescriptionKey: "Failed to sign token"]
            let error = NSError(domain: "ExampleApp", code: 0, userInfo: userInfo)
            failure(error)
        }
    }
    
    private func signCustomData(inLocal payload: [AnyHashable : Any]?, success: @escaping (String?) -> Void, failure: @escaping (Error?) -> Void) {
        let customData: UJETCustomData? = unsignedCustomData()
        
        let timestamp = Int(Date().timeIntervalSince1970)
        var data: [String : Any]? = nil
        if let aData = customData?.getData() {
            data = [
                "custom_data": aData,
                "iat": Int32(timestamp) /* required */,
                "exp": Int32(timestamp + 600) /* required */]
        }
        
        if let signedToken = encodeJWT(data) {
            success(signedToken)
        } else {
            let userInfo = [NSLocalizedDescriptionKey: "Failed to sign token"]
            let error = NSError(domain: "ExampleApp", code: 0, userInfo: userInfo)
            failure(error)
        }
    }
    
    private func signData(inRemote payload: [AnyHashable : Any]?, success: @escaping (String?) -> Void, failure: @escaping (Error?) -> Void) {
        let sessionConfiguration = URLSessionConfiguration.default
        let session = URLSession(configuration: sessionConfiguration)
        
        var urlRequest = URLRequest(url: URL(string: "https://your.company.com/api/ujet/sign")!)
        urlRequest.httpMethod = "POST"
        urlRequest.httpBody = try! JSONSerialization.data(withJSONObject: payload!, options: [])
        
        let task: URLSessionDataTask = session.dataTask(with: urlRequest, completionHandler: { data, response, error in
            if error != nil {
                failure(error)
                return
            }
            
            var json: [AnyHashable : Any]? = nil
            if let aData = data {
                json = try! JSONSerialization.jsonObject(with: aData, options: []) as? [AnyHashable : Any]
            }
            success((json?["token"] as! String))
        })
        
        task.resume()
    }
    
    private func encodeJWT(_ payload: [AnyHashable : Any]?) -> String? {
        // https://github.com/yourkarma/JWT/issues/204 Method chaning is not working in Xcode 10
        let algorithm: JWTAlgorithm? = JWTAlgorithmHSBase.algorithm384()
        let e = JWTBuilder.encodePayload(payload)!
        let s = e.secret(Bundle.main.infoDictionary?["UJETCompanySecret"] as? String)!
        let b = s.algorithm(algorithm)
        return b?.encode
    }
}
