//
//  UJETObject+CustomData.swift
//  ExampleApp
//
//  Copyright © UJET. All rights reserved.
//

import CoreBluetooth
import CoreLocation
import UIKit
import UJETKit

extension UJETObject {
    public func initCustomData() {
        if [.authorizedWhenInUse, .authorizedAlways].contains(CLLocationManager.authorizationStatus()) {
            _ = locationManager
        }
        
        if #available(iOS 13.0, *) {
            if CBCentralManager().authorization == .allowedAlways {
                _ = bluetoothManager
            }
        }
    }
    
    public func unsignedCustomData() -> UJETCustomData? {
        let customData = UJETCustomData()
        customData.set("os_version", label: "OS Version", stringValue: UIDevice.current.systemVersion)
        customData.set("model", label: "Model number", numberValue: 1234)
        customData.set("temperature", label: "Temperature", numberValue: 70.5)
        customData.set("purchase_date", label: "Purchase Date", dateValue: Date())
        customData.set("dashboard_url", label: "Dashboard", urlValue: URL(string: "http://internal.dashboard.com/1234"))
        customData.set("location", label: "Location", stringValue: location ?? "Location not available")
        customData.set("membership_number", label: "Membership Number", numberValue: (10000 + arc4random_uniform(100000) as NSNumber))
        customData.set("last_transaction_id", label: "Last Transaction ID", stringValue: UUID().uuidString)
        customData.set("bluetooth", label: "Bluetooth", stringValue: bluetoothState ?? "Bluetooth state unknown")
        
        return customData
    }
}

// MARK: CBCentralManagerDelegate
extension UJETObject: CBCentralManagerDelegate {

    
    public func centralManagerDidUpdateState(_ bluetoothManager: CBCentralManager) {
        switch bluetoothManager.state {
        case .resetting:
            bluetoothState = "Bluetooth resetting"
        case .unsupported:
            bluetoothState = "Bluetooth not supported"
        case .unauthorized:
            bluetoothState = "Bluetooth not authorized"
        case .poweredOff:
            bluetoothState = "Bluetooth is currently powered off"
        case .poweredOn:
            bluetoothState = "Bluetooth is currently powered on"
        default:
            bluetoothState = "Bluetooth state unknown"
        }
    }
}

// MARK: CLLocationManagerDelegate
extension UJETObject: CLLocationManagerDelegate {
    public func locationManager(_ manager: CLLocationManager, didChangeAuthorization status: CLAuthorizationStatus) {
        if status == .authorizedWhenInUse {
            manager.startUpdatingLocation()
        }
    }
    
    public func locationManager(_ manager: CLLocationManager, didUpdateLocations locations: [CLLocation]) {
        let clLocation = locations.last
        let geoCoder = CLGeocoder()
        if let aLocation = clLocation {
            geoCoder.reverseGeocodeLocation(aLocation, completionHandler: { placemarks, error in
                if let placemark = placemarks?.first {
                    self.location = [
                        placemark.locality,
                        placemark.administrativeArea,
                        placemark.country
                        ].compactMap({$0}).joined(separator: ", ")
                }
            })
        }
    }
    
    public func locationManager(_ manager: CLLocationManager, didFailWithError error: Error) {
        print("error: \(error.localizedDescription)")
    }
}
