//
//  AppDelegate.swift
//  ExampleApp
//
//  Copyright © UJET. All rights reserved.
//

import UIKit
import PushKit
import UserNotifications
import UJETKit

@UIApplicationMain
class AppDelegate: UIResponder, UIApplicationDelegate {

    var window: UIWindow?
    
    var ujetDelegate: UJETObject = UJETObject()


    func application(_ application: UIApplication, didFinishLaunchingWithOptions launchOptions: [UIApplication.LaunchOptionsKey: Any]?) -> Bool {
        
        UserDefaults.standard.set([
            "identifier": "user id",    // optional
            "name": "user name",        // optional
            "email": "user@example.com" // optional
            ], forKey: "user-data")
        
        let companyKey = Bundle.main.infoDictionary?["UJETCompanyKey"] as? String
        let subdomain = Bundle.main.infoDictionary?["UJETSubdomain"] as? String
        UJET.initialize(companyKey, subdomain: subdomain, delegate: ujetDelegate)
        UJET.setLogLevel(.verbose)
        
        let voipRegistry = PKPushRegistry(queue: DispatchQueue.main)
        voipRegistry.desiredPushTypes = Set([PKPushType.voIP])
        voipRegistry.delegate = self
        
        // in case you are using UserNotifications.framework
        UNUserNotificationCenter.current().delegate = self
        
        return true
    }
    
    func application(_ app: UIApplication, open url: URL, options: [UIApplication.OpenURLOptionsKey : Any] = [:]) -> Bool {
        if (url.scheme == "ujet") {
            return ujetDelegate.handleRouting(url)
        }
        
        return false

    }
    
    func application(_ application: UIApplication, continue userActivity: NSUserActivity, restorationHandler: @escaping ([UIUserActivityRestoring]?) -> Void) -> Bool {
        // Universal links
        if (NSUserActivityTypeBrowsingWeb == userActivity.activityType) {
            return ujetDelegate.handleRouting(userActivity.webpageURL!)
            
        } else if (userActivity.activityType == "INStartAudioCallIntent") {
            // Open app from Call history
            UJET.start(with: nil)
            
            return true
        }
        
        return false

    }
    
    func tokenFromData(data: Data) -> String {
        return data.map { String(format: "%02x", $0) }.joined()
    }
    
    func application(_ application: UIApplication, didRegisterForRemoteNotificationsWithDeviceToken deviceToken: Data) {
        print("apns token: ", tokenFromData(data: deviceToken))
        
        UJET.updatePushToken(deviceToken, type: .APN)
    }
    
    func application(_ application: UIApplication, didFailToRegisterForRemoteNotificationsWithError error: Error) {
        UJET.updatePushToken(nil, type: .APN)
    }
    
    func application(_ application: UIApplication, didReceiveRemoteNotification userInfo: [AnyHashable : Any], fetchCompletionHandler completionHandler: @escaping (UIBackgroundFetchResult) -> Void) {
        if (userInfo["ujet"] != nil) {
            UJET.receivedNotification(userInfo, completion: nil)
        }
    }
}

extension AppDelegate: PKPushRegistryDelegate {
    func pushRegistry(_ registry: PKPushRegistry, didUpdate credentials: PKPushCredentials, for type: PKPushType) {
        print("voip token: ", tokenFromData(data: credentials.token))
        
        if (type == .voIP) {
            UJET.updatePushToken(credentials.token, type: .voIP)
        }
    }
    
    func pushRegistry(_ registry: PKPushRegistry, didInvalidatePushTokenFor type: PKPushType) {
        if (type == .voIP) {
            UJET.updatePushToken(nil, type: .voIP)
        }
    }
    
    func pushRegistry(_ registry: PKPushRegistry, didReceiveIncomingPushWith payload: PKPushPayload, for type: PKPushType, completion: @escaping () -> Void) {
        if (type == .voIP && payload.dictionaryPayload["ujet"] != nil) {
            UJET.receivedNotification(payload.dictionaryPayload, completion: completion)
        } else {
            completion()
        }
    }
}

extension AppDelegate: UNUserNotificationCenterDelegate {
    // handle push received in foreground state
    func userNotificationCenter(_ center: UNUserNotificationCenter, willPresent notification: UNNotification, withCompletionHandler completionHandler: @escaping (UNNotificationPresentationOptions) -> Void) {
        let userInfo = notification.request.content.userInfo

        if (userInfo["ujet"] != nil) {
            UJET.receivedNotification(userInfo, completion: nil)
            
        } else {
            // handle yours
        }
    }
    
    // handle push received and tapped in background state
    func userNotificationCenter(_ center: UNUserNotificationCenter, didReceive response: UNNotificationResponse, withCompletionHandler completionHandler: @escaping () -> Void) {
        let userInfo = response.notification.request.content.userInfo

        if (userInfo["ujet"] != nil) {
            UJET.receivedNotification(userInfo, completion: nil)
            
        } else {
            // handle yours
        }
    }
}
