//
//  UJETObject.m
//  ExampleApp
//
//  Copyright © UJET. All rights reserved.
//

#import "UJETObject.h"
#import "UJETObject+CustomData.h"

@import JWT;

@implementation UJETObject

#pragma mark - Delegates

#pragma mark UJETDelegate

- (void)signPayload:(NSDictionary *)payload payloadType:(UjetPayloadType)payloadType success:(void (^)(NSString *))success failure:(void (^)(NSError *))failure {
    /**
     * We strongly recommend to sign the payload from your application server not in the client
     * This exmaple uses local signing for testing purpose. Use below in production
     * [self signDataInRemote:payload success:success failure:failure];
     */
    
    if (payloadType == UjetPayloadAuthToken) {
        [self signAuthTokenInLocal:payload success:success failure:failure];
        
    } else if (payloadType == UjetPayloadCustomData) {
        [self signCustomDataInLocal:payload success:success failure:failure];
    }
}

#pragma mark - Private

- (void)signAuthTokenInLocal:(NSDictionary *)payload success:(void (^)(NSString *))success failure:(void (^)(NSError *))failure {
    NSMutableDictionary *payloadData = [payload mutableCopy];
    
    NSDictionary *userData = [[NSUserDefaults standardUserDefaults] objectForKey:@"user-data"];
    [payloadData addEntriesFromDictionary:userData];
    payloadData[@"iat"] = [NSNumber numberWithDouble:[[NSDate date] timeIntervalSince1970]]; // required
    payloadData[@"exp"] = [NSNumber numberWithDouble:([[NSDate date] timeIntervalSince1970] + 600)]; // required
    
    NSString *signedToken = [self encodeJWT:payloadData];
    
    if (signedToken.length > 0) {
        success(signedToken);
        
    } else {
        NSDictionary *userInfo = @{ NSLocalizedDescriptionKey: @"Failed to sign token" };
        NSError *error = [NSError errorWithDomain:@"ExampleApp" code:0 userInfo:userInfo];
        failure(error);
    }
}

- (void)signCustomDataInLocal:(NSDictionary *)payload success:(void (^)(NSString *))success failure:(void (^)(NSError *))failure {
    /**
     * If you want to pass unsigned custom data, please put `success(nil);` here and
     * pass the custom data when starting the SDK (see ViewController.m).
     */
    
    UJETCustomData *customData = [self unsignedCustomData];
    
    // `iat`, and `exp` must be included
    int timestamp = (int)[[NSDate date] timeIntervalSince1970];
    NSDictionary *data = @{
                           @"custom_data": [customData getData],
                           @"iat": [NSNumber numberWithInt:timestamp], // required
                           @"exp": [NSNumber numberWithInt:timestamp + 600] // required
                           };
    
    NSString *signedToken = [self encodeJWT:data];
    
    if (signedToken.length > 0) {
        success(signedToken);
        
    } else {
        NSDictionary *userInfo = @{ NSLocalizedDescriptionKey: @"Failed to sign token" };
        NSError *error = [NSError errorWithDomain:@"ExampleApp" code:0 userInfo:userInfo];
        failure(error);
    }
}

- (void)signDataInRemote:(NSDictionary *)payload success:(void (^)(NSString *))success failure:(void (^)(NSError *))failure {
    NSURLSessionConfiguration *sessionConfiguration = [NSURLSessionConfiguration defaultSessionConfiguration];
    NSURLSession *session = [NSURLSession sessionWithConfiguration:sessionConfiguration];
    
    NSMutableURLRequest *mutableRequest = [[NSMutableURLRequest alloc] init];
    mutableRequest.URL = [NSURL URLWithString:@"https://your.company.com/api/ujet/sign"];
    mutableRequest.HTTPMethod = @"POST";
    mutableRequest.HTTPBody = [NSJSONSerialization dataWithJSONObject:payload options:0 error:nil];
    
    NSURLSessionDataTask *task = [session dataTaskWithRequest:mutableRequest completionHandler:^(NSData *data, NSURLResponse *response, NSError *error) {
        if (error) {
            failure(error);
            return;
        }
        
        NSDictionary *json = [NSJSONSerialization JSONObjectWithData:data options:0 error:nil];
        success(json[@"token"]);
    }];
    
    [task resume];
}

- (NSString *)encodeJWT:(NSDictionary *)payload {
    id<JWTAlgorithm> algorithm = [JWTAlgorithmHSBase algorithm384];
    NSString *secret = NSBundle.mainBundle.infoDictionary[@"UJETCompanySecret"];
    return [JWTBuilder encodePayload:payload].secret(secret).algorithm(algorithm).encode;
}

@end
