//
//  UJETObject+CustomData.m
//  ExampleApp
//
//  Copyright © UJET. All rights reserved.
//

#import "UJETObject+CustomData.h"

@import SystemConfiguration.CaptiveNetwork;

@implementation UJETObject (CustomData)

- (void)initCustomData {
    [self initLocationManager];
    [self initBluetoothManager];
}

- (UJETCustomData *)unsignedCustomData {
    UJETCustomData *customData = [UJETCustomData new];
    [customData set:@"os_version" label:@"OS Version" stringValue:[[UIDevice currentDevice] systemVersion]];
    [customData set:@"model" label:@"Model number" numberValue:[NSNumber numberWithInteger:1234]];
    [customData set:@"temperature" label:@"Temperature" numberValue:[NSNumber numberWithFloat:70.5]];
    [customData set:@"purchase_date" label:@"Purchase Date" dateValue:[NSDate date]];
    [customData set:@"dashboard_url" label:@"Dashboard" urlValue:[NSURL URLWithString:@"http://internal.dashboard.com/1234"]];
    [customData set:@"location" label:@"Location" stringValue:(self.location.length > 0) ? self.location : @"Location not available"];
    [customData set:@"membership_number" label:@"Membership Number" numberValue:@(10000+arc4random_uniform(100000))];
    [customData set:@"last_transaction_id" label:@"Last Transaction ID" stringValue:[[NSUUID UUID] UUIDString]];
    
    NSString *wifiName = [self getWifiSSID];
    [customData set:@"wifi" label:@"Wi-Fi" stringValue:(wifiName.length > 0) ? wifiName : @"Wi-Fi not connected"];
    [customData set:@"bluetooth" label:@"Bluetooth" stringValue:(self.bluetoothState.length > 0) ? self.bluetoothState : @"Bluetooth state unknown"];
    
    return customData;
}

#pragma mark - Delegates

#pragma mark CBCentralManagerDelegate

- (void)centralManagerDidUpdateState:(nonnull CBCentralManager *)bluetoothManager {
    switch(bluetoothManager.state)
    {
        case CBManagerStateResetting: self.bluetoothState = @"Bluetooth resetting"; break;
        case CBManagerStateUnsupported: self.bluetoothState = @"Bluetooth not supported"; break;
        case CBManagerStateUnauthorized: self.bluetoothState = @"Bluetooth not authorized"; break;
        case CBManagerStatePoweredOff: self.bluetoothState = @"Bluetooth is currently powered off"; break;
        case CBManagerStatePoweredOn: self.bluetoothState = @"Bluetooth is currently powered on"; break;
        default: self.bluetoothState = @"Bluetooth state unknown"; break;
    }
}

#pragma mark CLLocationManagerDelegate

- (void)locationManager:(CLLocationManager *)manager didChangeAuthorizationStatus:(CLAuthorizationStatus)status {
    if (status == kCLAuthorizationStatusAuthorizedWhenInUse) {
        [manager startUpdatingLocation];
    }
}

- (void)locationManager:(CLLocationManager *)manager didUpdateLocations:(NSArray *)locations {
    CLLocation* clLocation = [locations lastObject];
    CLGeocoder * geoCoder = [CLGeocoder new];
    [geoCoder reverseGeocodeLocation:clLocation completionHandler:^(NSArray<CLPlacemark *> * _Nullable placemarks, NSError * _Nullable error) {
        if (placemarks.count > 0) {
            self.location = [NSString stringWithFormat:@"%@, %@, %@", placemarks[0].locality, placemarks[0].administrativeArea, placemarks[0].country];
        }
    }];
}

- (void)locationManager:(CLLocationManager *)manager didFailWithError:(NSError *)error {
    NSLog(@"error: %@", error.localizedDescription);
}

#pragma mark - Private

- (NSString *)getWifiSSID {
    NSArray *interFaceNames = (__bridge_transfer id)CNCopySupportedInterfaces();
    
    for (NSString *name in interFaceNames) {
        NSDictionary *info = (__bridge_transfer id)CNCopyCurrentNetworkInfo((__bridge CFStringRef)name);
        
        if (info[@"SSID"]) {
            return info[@"SSID"];
        }
    }
    return nil;
}

- (void)initLocationManager {
    if (CLLocationManager.authorizationStatus < kCLAuthorizationStatusAuthorizedAlways) {
        return;
    }
    
    self.locationManager = [CLLocationManager new];
    CLLocationManager *locationManager = (CLLocationManager *)self.locationManager;
    locationManager.delegate = self;
    locationManager.desiredAccuracy = kCLLocationAccuracyNearestTenMeters;
    locationManager.distanceFilter = 500;
    [locationManager startUpdatingLocation];
}

- (void)initBluetoothManager {
    if (@available (iOS 13, *)) {
        if (CBCentralManager.new.authorization != CBManagerAuthorizationAllowedAlways) {
            return;
        }
    }
    
    NSDictionary *options = @{ CBCentralManagerOptionShowPowerAlertKey: @0 };
    
    self.bluetoothManager= [[CBCentralManager alloc] initWithDelegate:self
                                                                queue:nil
                                                              options:options];
}

@end
