%
% function h = smith(S, lStyle);
%
% Draws data into a smith chart. If the largest absolute value
% of the reflexion coefficients is above 1, the circle |r|=1
% is plotted thickly and the radius of the smith chart is written
% into the lower left corner of the figure. If more than one
% curve should be drawn into one diagramm, one can use an
% S-parameter matrix or the command "hold on".
%
% Parameters:  S      - complex reflexion coefficients (matrixes
%                       are allowed.)
%              lStyle - (optional) colour and shape of the curve
%                       (see 'help plot')
%              h      - (optional) handle of the figure
%
% Microwave Toolbox 2.6
% M. Margraf, margraf@mwt.ee.tu-berlin.de
% J. Horn, horn@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function h = smith(S, lStyle);

if (nargin<1) | (nargin>2)
    error('Two or three input parameters are required!');
end
if ischar(S)
    error('First input parameter has to be numerical!');
end
if nargin==2 if ~ischar(lStyle)
    error('Second input parameter has to be a character string!');
end; end
if isempty(S)
    error('S-parameter variable is empty!');
end

rMAX = max(max(1, max(abs(S))));     % Einheitskreis oder grer
if rMAX > 1
    rMAX = 1.05*rMAX;   % etwas grer als ntig zeichnen
end
rMAXq = rMAX*rMAX; 

nn   = 50;      % Anzahl der Punkte pro Kreis
sm   = 3;       % Anzahl der Kreisbgen mit Im(z)=const
theta = 0:2*pi/nn:2*pi;       % Winkel fr Vollkreis


if ishold
    boolLeaveWithHold=1;
    set(gca,'NextPlot','add')% (obsolet?)
    cax   = newplot;
    txtC  = get(cax,'xcolor');      % Text-Farbe
    txtB  = get(cax, 'color');      % Hintergrund-Farbe
    ChartHandles=get(cax,'UserData');% smitch-chart patch- und line-handles
    ChartPatchhandle=findall(ChartHandles,'flat','type','patch');
    ChartTexthandle=findall(ChartHandles,'flat','type','text');
    ChartLinehandles=findall(ChartHandles,'flat','type','line');
    
    % greres Chart notwendig?
    if rMAX <= max(get(ChartPatchhandle,'xdata'))
        boolPlotNewChart=0;
    else
        boolPlotNewChart=1;
        if exist('ChartTexthandle','var'),delete(ChartTexthandle),end% alten Radius-Text lschen
        delete(ChartLinehandles)% alte Kreise lschen
        set(ChartPatchhandle,'xdata',rMAX*cos(theta), 'ydata',rMAX*sin(theta))% Patch vergrern
    end
else
    boolPlotNewChart=1;
    boolLeaveWithHold=0;
    set(gca,'NextPlot','replace') %(obsolet?)
    cax   = newplot;
    txtC  = get(cax,'xcolor');      % Text-Farbe
    txtB  = get(cax, 'color');      % Hintergrund-Farbe
    %clf             % Diagramm lschen 
    axis('equal')   % Seitenverhltnis 1:1
    axis('off')     % Achsen ausschalten  
    
    ChartPatchhandle=patch('xdata',rMAX*cos(theta), 'ydata',rMAX*sin(theta),...
                           'edgecolor',txtC,'facecolor',txtB, 'HandleVisibility', 'off');  % uersten Kreis zeichnen 
end

if boolPlotNewChart       % neues Diagramm zeichnen ?
    hold on
    linecount=0;
    clear ChartLinehandles
    
    Kreis = exp(j*theta);
    if rMAX > 1
        linecount=linecount+1;
        ChartLinehandles(linecount,1)=plot(Kreis, 'color',txtC,'linewidth',2, 'HandleVisibility', 'off');   % Kreis mit |r|=1 fett zeichnen
    end
    
    linecount=linecount+1;
    ChartLinehandles(linecount,1)=plot( [-rMAX rMAX], [0 0], 'color',txtC,'linewidth',1,...
                                        'HandleVisibility', 'off');   % x-Achse von r
    linecount=linecount+1;
    ChartLinehandles(linecount,1)=plot( [1 1], sqrt(rMAXq-1)*[1 -1], 'color',txtC,'linewidth',1 ,...
                                       'HandleVisibility', 'off'); % falls ntig senkrechte Linie durch Leerlauf zeichnen
    
    % **********************************************************************************
    % Kreisbgen mit Im(z)=const
    for m=1:sm
        r  = exp(j*pi*m/(sm+1));     % Kreisbgen gleichmig auf |r|=1 verteilen
        Im = imag((1+r)/(1-r)) / (rMAX^0.7);    % Im(z)/sqrt(rMAX) => Kreisbgen gleichmig verteilen
        Mitte  = 1 + j/Im;
        Radius = 1 / abs(Im);
        
        if rMAXq == 1       % Smith-Chart mit |r|=1
            theta = 1.5*pi;
            beta  = angle(r - Mitte);
        else                % Smith-Chart mit |r|>1
            Reell  = (rMAXq+1)/(rMAXq-1);
            Wurzel =  Reell*Reell - Im*Im-1;
            if Wurzel<0     % liegt Kreis vollkommen in Smith-Chart ?
                beta = 0;       % ja, ...
                theta = 2*pi;   % ... dann Vollkreis zeichnen
            else
                Reell1 =  sqrt(Wurzel)-Reell;   % beide Schnittpunkte mit Auenkreis berechnen
                Reell2 = -sqrt(Wurzel)-Reell;
                
                r = (Reell1+j*Im-1) / (Reell1+j*Im+1);
                beta = angle(r - Mitte);        % Anfangswinkel
                r = (Reell2+j*Im-1) / (Reell2+j*Im+1);
                theta = angle(r - Mitte);       % Endwinkel
            end
        end
        
        if beta<0
            beta = 2*pi + beta;   % Wertebereich 0...2*pi
        end
        if (theta<0) | (theta<beta)
            theta = 2*pi + theta;   % Wertebereich anpassen (evtl. >2*pi)
        end
        
        Kreis = Mitte + Radius*exp(j*linspace(beta,theta,nn));  % Punkte des Kreises berechnen
        linecount=linecount+1;
        ChartLinehandles(linecount,1)=plot(Kreis, 'color',txtC,'linewidth',1,...
                                         'HandleVisibility', 'off');     % obere Kreisbgen zeichnen
        
        Kreis = real(Kreis) - j*imag(Kreis);        % Kreisbgen an reeller Achse spiegeln
        linecount=linecount+1;
        ChartLinehandles(linecount,1)=plot(Kreis, 'color',txtC,'linewidth',1,...
                                         'HandleVisibility', 'off');     % untere Kreisbgen zeichnen
    end
    
    % **********************************************************************************
    % Kreise mit Re(z)=const zeichnen
    for m=1:sm
        r = m*(rMAX+1)/(sm+1) - rMAX;   % Kreise gleichmig verteilen
        z = real((1+r) / (1-r));
        Radius = 1 / (z+1);
        Mitte  = z * Radius;
        Kreis = Mitte + Radius*exp(j*2*pi*(0:nn)/nn);
        if abs(abs(r)-1) > 0.4      % sieht unschn aus (zu nah an |r|=1) ?
            linecount=linecount+1;
            ChartLinehandles(linecount,1)=plot(Kreis, 'color',txtC,'linewidth',1,...
                                                'HandleVisibility', 'off');     % linke Kreisbgen zeichnen
        end
        
        if rMAX > 1     % rechte Kreisbgen zeichnen ?
            r     = Mitte + 3*Radius;
            Reell = (1+r)/(1-r);
            Im    = - Reell*Reell - 2*Reell*(rMAXq+1)/(rMAXq-1) - 1;
            if Im <= 0      % liegt Kreis vollstndig im Smith-Chart ?
                Kreis = 2 - Kreis;    % ja, dann Kreis an r=1 spiegeln
            else
                r = (Reell+j*sqrt(Im)-1) / (Reell+j*sqrt(Im)+1);
                beta = angle(r - 2+Mitte);        % Anfangswinkel
                Kreis = 2-Mitte + Radius*exp(j*linspace(beta,2*pi-beta,nn));
            end
            linecount=linecount+1;
            ChartLinehandles(linecount,1)=plot(Kreis, 'color',txtC,'linewidth',1,...
                                                'HandleVisibility', 'off');     % Kreis zeichnen
        end
    end
    
    if rMAX>1
        ChartTexthandle=text(-rMAX, -rMAX, sprintf('r = %0.3g', rMAX), 'verticalalignment','bottom',...
                                'HandleVisibility', 'off');
    end
    if exist('ChartTexthandle','var')&~isempty(ChartTexthandle)
        set(cax,'UserData',[ChartLinehandles(1:linecount,1); ChartPatchhandle; ChartTexthandle])
    else
        set(cax,'UserData',[ChartLinehandles(1:linecount,1); ChartPatchhandle])
    end
    
end

% **********************************************************************************
if nargin == 2
    hh = plot(real(S), imag(S), lStyle);    % zeichne Daten
else
    hh = plot(real(S), imag(S));            % zeichne Daten
end
if nargout > 0      % Handle zurckgeben ?
    h = hh;
end

if ~boolLeaveWithHold
    hold off
end
