%
% function S = chain(Sa, Sb);
%
% Calculates the chain connection of two 2-port networks, i.e.
% connecting one after another.
%
% Parameters:  Sa  -  S-parameter of the first 2-port network
%              Sb  -  S-parameter of the second 2-port network
%              S   -  S-parameter of the chain network
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function S = chain(Sa, Sb);

if nargin ~= 2
    error('Two input parameters have to be delivered!');
end
if ischar(Sa) | ischar(Sb)
    error('Input parameters have to be numerical!');
end
[m1, n1] = size(Sa);
[m2, n2] = size(Sb);
if (n1~=4) | (n2~=4)
    error('2-port S-parameters have to be delivered!');
end

n      = 1 - Sb(:,1).*Sa(:,4);
m      = (n ~= 0);    % memorize special cases
S(m,1) = Sa(m,1) + Sa(m,2).*Sa(m,3).*Sb(m,1)./n(m,1);  % S11
S(m,2) = Sa(m,2).*Sb(m,2)./n(m,1);                     % S12
S(m,3) = Sa(m,3).*Sb(m,3)./n(m,1);                     % S21
S(m,4) = Sb(m,4) + Sb(m,2).*Sb(m,3).*Sa(m,4)./n(m,1);  % S22

S(~m,1) = inf;  % S11
S(~m,2) = inf;  % S12
S(~m,3) = inf;  % S21
S(~m,4) = inf;  % S22
