%
% function S = Z2S(Z, Z0);
%
% Transforms resistance parameters (Z-parameters) of a 2-port network
% into S-parameters.
%
% Parameters:  Z  - Z-parameters of a 2-port network
%              Z0 - (optional) reference impedance of the S-parameters;
%                   default 50 ohms
%              S  - S-parameters of the 2-port network normalized to Z0
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function S = Z2S(Z, Z0);

if (nargin < 1) | (nargin > 2)
    error('One or two input parameters are required!');
end
if ischar(Z)
    error('First input parameter has to be numerical!');
end
[m,n] = size(Z);
if n~=4
    error('Number of columns of the first input parameter has to be four!');
end

if nargin == 1
    Z0 = 50;	% Default-Systemimpedanz ist 50 Ohm
else
    if ischar(Z0)
        error('Second input parameter has to be numerical!');
    elseif length(Z0) ~= 1
        error('Second input parameter has to be scalar!');
    end
end

Z      = Z/Z0;  % Z-Parameter normieren
Nenner = (1+Z(:,1)).*(1+Z(:,4)) - Z(:,2).*Z(:,3);

S(:,1) = ((Z(:,1)-1).*(Z(:,4)+1) - Z(:,2).*Z(:,3)) ./ Nenner;
S(:,2) = 2*Z(:,2) ./ Nenner;
S(:,3) = 2*Z(:,3) ./ Nenner;
S(:,4) = ((Z(:,1)+1).*(Z(:,4)-1) - Z(:,2).*Z(:,3)) ./ Nenner;
