%
% function S = Y2S(Y, Y0);
%
% Transforms conductance parameters (Y-parameters) of a 2-port
% networks into S-parameters.
%
% Parameters:  Y  - Y-parameters of a 2-port network
%              Y0 - (optional) reference admittance of the
%                   S-parameters; default 20 mS
%              S  - S-parameters of the 2-port network normalized
%                   to Y0
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function S = Y2S(Y, Y0);

if (nargin < 1) | (nargin > 2)
    error('One or two input parameters are required!');
end
if ischar(Y)
    error('First input parameter has to be numerical!');
end
[m,n] = size(Y);
if n~=4
    error('Numer of columns of the first parameter has to be four!');
end

if nargin == 1
    Y0 = 2e-2;	% Default-Systemadmittanz ist 20 mS
else
    if ischar(Y0)
        error('Second input parameter has to be numerical!');
    elseif length(Y0) ~= 1
        error('Second input parameter has to be scalar!');
    end
end

Y      = Y/Y0;  % Y-Parameter normieren
Nenner = (1+Y(:,1)).*(1+Y(:,4)) - Y(:,2).*Y(:,3);

S(:,1) = ((1-Y(:,1)).*(1+Y(:,4)) + Y(:,2).*Y(:,3)) ./ Nenner;
S(:,2) = -2*Y(:,2) ./ Nenner;
S(:,3) = -2*Y(:,3) ./ Nenner;
S(:,4) = ((1+Y(:,1)).*(1-Y(:,4)) + Y(:,2).*Y(:,3)) ./ Nenner;
