%
% function [SCircle, Region] = StabCircleL(S, nn);
%
% Calculates stability circle of the load. If desired, the region
% of stability is also calculated. ('i' = inside the circle,
% 'o' = outside the circle)
%
% Parameters: S       -  S-parameters of a 2-port network
%             nn      -  (optional) number of points per circle;
%                        default 51
%             SCircle -  stability circle of the load
%             Region  -  region of stability (inside/outside)
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function [SCircle, Region] = StabCircleL(S, nn);

if (nargin<1) | (nargin>2)
    error('One or two input parameters are required!');
end
if ischar(S)
    error('First input parameter has to be numerical!');
end
[m, n] = size(S);
if n ~= 4
    error('First input parameter has to obtain 4 columns!');
end
if nargin == 1
    nn = 51;
else
    if ischar(nn)
        error('Second input parameter has to be numerical!');
    elseif length(nn) ~= 1
        error('Second input parameter has to be scalar!');
    end
end

detS   = S(:,1).*S(:,4) - S(:,2).*S(:,3); % Determinante
Mitte  = (conj(detS).*S(:,1) - conj(S(:,4))) ./ (abs(detS).^2 - abs(S(:,4)).^2);
Radius = abs(S(:,2).*S(:,3) ./ (abs(detS).^2 - abs(S(:,4)).^2));

K = exp(j*linspace(0,2*pi,nn))';  % Einheitskreis mit n Punkten
for k=1:m  % fr jede Frequenz einen Kreis berechnen
    Kreis(:,k) = Mitte(k) + Radius(k) * K;
end

if nargout > 1  % Stabilittsregionen berechnen ?
    Region(1:m) = 'o';
    m1  = abs(Mitte)  < Radius;
    m2  = abs(S(:,1)) < 1;
    Region( m1 & m2) = 'i';
    Region(~(m1|m2)) = 'i';
end
