%
% function S_Z1 = Snorm(S_Z0, Z0, Z1);
%
% Changes the reference impedance Z0 of the S-parameters S_Z0
% to Z1. If Z0 and/or Z1 is not equal for all ports, a vector
% has to be delivered containing the reference impedance of
% every port. If a scalar reference impedance is delivered,
% it holds for all ports.
%
% Parameters:  S_Z0 -  S-parameters normalized to Z0
%              Z0   -  reference impedance of the input S-parameters
%                      (vector or scalar)
%              Z1   -  reference impedance of the output S-parameters
%                      (vector or scalar)
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function S_Z1 = Snorm(S_Z0, Z0, Z1);

if nargin ~= 3
    error('Three input parameters are required!');
end
if ischar(S_Z0) | ischar(Z0) | ischar(Z1)
    error('Input parameters has to be numerical!');
end
[m, n] = size(S_Z0);
q      = sqrt(n);
if mod(q,1) ~= 0
    error('Number of columns of the first input parameters has to be square!');
end
if (length(Z0)~=1) & (length(Z0)~=q)
    error('Wrong length of the second input parameter!');
end
if (length(Z1)~=1) & (length(Z1)~=q)
    error('Wrong length of the third input parameter!');
end
if any(Z0<=0) | any(Z1<=0)
    error('Normalizing to less or equal 0 ohm does not make sense!');
end

L0 = length(Z0);
if L0==1            % falls fr alle Ports gleich ...
    Z0(1:q) = Z0;   % ..., dann trotzdem Vektor erzeugen
end
L1 = length(Z1);
if L1 ==1           % falls fr alle Ports gleich ...
    Z1(1:q) = Z1;   % ..., dann trotzdem Vektor erzeugen
end
r = zeros(q);
for w=1:q       % Diagonal-Matrix erzeugen
    r(w,w) = (Z1(w)-Z0(w)) / (Z1(w)+Z0(w));  % formaler Reflexionsfaktor
end

for v=1:m
    for w=0:n-1   % quadratische S-Parameter-Matrix erstellen
        S(floor(w/q)+1, mod(w,q)+1) = S_Z0(v,w+1);
    end

    S = (S-r) * inv((eye(q)-r*S));   % S-Parameter umnormieren

    for w=0:q-1   % in bliche Matrixform (nicht quadratisch) schreiben
        S_Z1(v,w*q+1:w*q+q) = S(w+1,:);
    end
end

if (L0==1) & (L1==1) return; end    % alle Port-Impedanzen identisch, dann folgende Normung unntig
for w=0:n-1     % an jedem Port auf neue Impedanz normieren
    zeile  = floor(w/q)+1;
    spalte = mod(w,q)+1;
    S_Z1(:,w+1) = S_Z1(:,w+1) * sqrt(Z1(zeile)/Z0(zeile) * Z0(spalte)/Z1(spalte)) * (Z1(zeile)+Z0(zeile))/Z1(zeile) * Z1(spalte)/(Z1(spalte)+Z0(spalte));
end
