%
% function SaveSnP(FileName, freq, S, freqN, N);
%
% Generates a Touchstone file (SnP-file, e.g. FHC40.s2p). The
% S-parameter matrix has to contain one S-parameter per column:
% S11, S12, ..., S21, S22, ... . The S-parameters has to be
% normalized to 50 ohms. Noise parameters (if used) has to contain
% the following columns: minimum noise figure in decibel, optimum
% source reflexion coefficient normalized to 50 ohms, normalized
% effective noise resistance normalized to 50 ohms.
%
% Parameters:  FileName  -  name of the Touchstone file
%              freq      -  vector containing the frequency of the
%                           S-parameters
%              S         -  matrix containing complex S-parameters
%              freqN     -  optional: frequency of the noise parameters
%              N         -  optional: noise parameters
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function SaveSnP(FileName, freq, S, freqN, N);

if (nargin~=3) & (nargin~=5)
    error('Three or five input parameters are required!');
end
if ~ischar(FileName)
   error('First input parameter has to be a character string!');
end
if ischar(freq) | ischar(S)
   error('Second and third input parameter has to be numerical!');
end

[m, n] = size(freq);
if (m~=1) & (n~=1)
   error('Second input parameter has to be 1-dimensional!');
end
[m, n] = size(S);
if m ~= length(freq)
    error('Number of rows of the third input parameter must equal the size of the second one!');
end
if n > 100
    error('Maximum number of S-parameter ports is 10!');
end
if mod(sqrt(n),1) ~= 0
    error('Number of columns of the S-parameter matrix has to be 1,4,9,16,... (1,2,3,4-port network etc.)!');
end
if nargin == 5
    if ischar(N)
        error('Fourth input parameter has to be numerical!');
    elseif n ~= 4
        error('Noise parameters can only be used with 2-port networks!');
    end

    [u, v] = size(N);
    if v ~= 3
        error('Number of columns of the noise parameters has to be 3!');
    end
    if freq(length(freq)) < freqN
        error('First noise parameter frequency has to be less/equal than the last S-parameter frequency!');
    end
    if length(freqN) ~= u
        error('Number of rows of the fifth input parameter must equal the size of the fourth one!');
    end
end

h = fopen(FileName, 'w');      % Datei erstellen
fprintf(h, '! ------------------------------------------------------------------------\r\n');
fprintf(h, '! Erstellt mit MATLAB-Funktion von M. Margraf\r\n');
fprintf(h, '! Created  by  MATLAB function by  M. Margraf\r\n');
fprintf(h, '! ------------------------------------------------------------------------\r\n');
fprintf(h, '! symbol freq-unit parameter-type data-format keyword impedance-ohm\r\n');
fprintf(h, '#        HZ        S              MA          R       50\r\n');

if n == 1       % 1-Tor ?
    fprintf(h, '! freq    magS11   angS11\r\n');
elseif n == 4   % 2-Tor ?
    S = [S(:,1) S(:,3) S(:,2) S(:,4)];  % bei 2-Tor-Parametern S12 und S21 vertauschen
    fprintf(h, '! freq    magS11     angS11      magS21      angS21      magS12      angS12      magS22       angS22\r\n');
else
    fprintf(h, '! freq    magnitude    angle  ....\r\n');
end

for i=1:length(freq)            % jede Daten-Zeile einzeln schreiben
    fprintf(h, '%g  ', freq(i));
    for j=1:n
        fprintf(h, '%g %g ', abs(S(i,j)), angle(S(i,j))*180/pi);
        if n<=4
            if j == n           % bei kleinerer Tor-Anzahl alle S-Parameter in einer Zeile
                fprintf(h, '\r\n');   % Zeilen-Umbruch
            end
        elseif mod(j, sqrt(n)) == 0   % bei grerer Tor-Anzahl in mehrere Zeilen verteilen
            fprintf(h, '\r\n');       % Zeilen-Umbruch
        end
    end
end

if nargin == 5
fprintf(h, '! Noise data:  freq    Fmin     Gamma-opt     rn\r\n');
for i=1:u       % falls vorhanden, Rausch-Parameter schreiben
    % optimaler Quellenreflexionsfaktor wird immer linear angegeben
    fprintf(h, '%g  %g %g %g %g\r\n', freqN(i), N(i,1), abs(N(i,2)), angle(N(i,2))*180/pi, N(i,3));
end
end

fclose(h);      % Datei schlieen
