%
% function SaveCITI(FileName, indep, dep, names);
%
% Generates a CITI-file (Common Instrumentation Transfer and
% Interchange). The independent variable has to extend in one
% dimension. The dependent variable can extend into two dimensions.
% The names of the variables is exported into the CITI-file, if
% a variable (and not an expression) is delivered to the function.
% otherwise the variables are named "indepVar" and "depVar",
% respectivly. The names of the dependent variables can be delivered
% in a character string. This especially makes sense, if a
% two-dimensional dependent variable is used.
% Example: SaveCITI('data.cit', Ugs, Dat, ['n1';'n2';'nr3'])
%
% Parameters:  FileName   - name of the CITI-file
%              indep      - independent variable
%              dep        - dependent variable(s)
%              names      - (optional) names for the dependent
%                           variable(s)
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function SaveCITI(FileName, indep, dep, names);

if (nargin<3) | (nargin>4)
    error('Three or four input parameters are required!');
end
if ~ischar(FileName)
    error('First input parameter has to be a character string!');
end
if ischar(indep) | ischar(dep)
    error('Second and third input parameter has to be numerical!');
end

[mi, ni] = size(indep);
if ni > 1
    if mi == 1
        indep = conj(indep');     % in Zeilenvektor wandeln
        dep = conj(dep');        % in Zeilenvektor wandeln (der Einheit wegen)
    else
        error('Independent variable has to be one-dimensional!');
    end
end

[md, nd] = size(dep);
if nd > 1
    if md == 1
        dep = conj(dep');     % in Zeilenvektor wandeln
    end
end

[md, nd] = size(dep);
if md ~= length(indep)
    error('Length of the data vectors has to be equal!');
end

if nargin==4
    if ~ischar(names)
        error('Fourth input parameter has to be a character string!');
    end
    [ms, ns] = size(names);
    if nd ~= ms
        error('Wrong numer of names!');
    end
else
    if isempty(inputname(3))
        s = 'depVar';      % wenn ntig, Namen selber vergeben
    else
        s = inputname(3);
    end
    if nd>1
        for m=1:nd
            names(m,:) = sprintf('%s%i',s,m);
        end
    else
        names = s;
    end
end

% ************************************************************************************
fid = fopen(FileName, 'w');      % Datei erstellen
fprintf(fid, 'CITIFILE A.01.00\r\n');   % Kopfzeile schreiben
fprintf(fid, 'NAME DATA\r\n');

if ~isreal(indep)
    Typen = 'RI';       % Variable ist complex
else
    Typen = 'MAG';      % Variable ist reell
end

if isempty(inputname(2))    % wenn keine Variable bergeben wurde ...
    Name = 'indepVar';      % ... selber einen Namen vergeben
else
    Name = inputname(2);
end

fprintf(fid, 'VAR  %s %s %i\r\n', Name, Typen, length(indep));

if ~isreal(dep)
    Typen = 'RI';       % Variable ist complex
else
    Typen = 'MAG';      % Variable ist reell
end
    
for m=1:nd
    fprintf(fid, 'DATA %s %s %i\r\n', names(m,:), Typen, md);
end

% unabhngige Variable in Datei schreiben
fprintf(fid, 'VAR_LIST_BEGIN\r\n');
if isreal(indep)
    fprintf(fid, '%g\r\n',indep);
else
    for n=1:length(indep)
        fprintf(fid, '%g, %g\r\n',real(indep(n)), imag(indep(n)));
    end
end
fprintf(fid, 'VAR_LIST_END\r\n');

% abhngige Variable in Datei schreiben
for n=1:nd
    fprintf(fid, 'BEGIN\r\n');
    if isreal(dep)
        fprintf(fid, '%g\r\n',dep(:,n));
    else
        for m=1:md
            fprintf(fid, '%g, %g\r\n',real(dep(m,n)), imag(dep(m,n)));
        end
    end
    fprintf(fid, 'END\r\n');
end

fclose(fid);
