%
% function Y = S2Y(S, Y0);
%
% Transforms S-parameters of a 2-port network into admittance parameters
% (Y-parameters).
%
% Parameters:  S  - S-parameters of the 2-port network normalized to Y0
%              Y0 - (optional) reference admittance of the S-parameters;
%                   default 20 mS
%              Y  - Y-parameters of the 2-port network
%              
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function Y = S2Y(S, Y0);

if (nargin < 1) | (nargin > 2)
    error('One or two input parameters are required!');
end
if ischar(S)
    error('First input parameter has to be numerical!');
end
[m,n] = size(S);
if n~=4
    error('Number of columns of the first input parameter has to be four!');
end

if nargin == 1
    Y0 = 2e-2;	% Default-Systemadmittanz ist 20 mS
else
    if ischar(Y0)
        error('Second input parameter has to be numerical!');
    elseif length(Y0) ~= 1
        error('Second input parameter has to be scalar!');
    end
end

Nenner = (1+S(:,1)).*(1+S(:,4)) - S(:,2).*S(:,3);

Y(:,1) = ((1-S(:,1)).*(1+S(:,4)) + S(:,2).*S(:,3)) ./ Nenner;
Y(:,2) = -2*S(:,2) ./ Nenner;
Y(:,3) = -2*S(:,3) ./ Nenner;
Y(:,4) = ((1+S(:,1)).*(1-S(:,4)) + S(:,2).*S(:,3)) ./ Nenner;

Y = Y*Y0;  % Y-Parameter de-normieren
