%
% function H = S2H(S, Z0);
%
% Transforms S-parameters of a 2-port network into hybrid parameters
% (H-parameters).
%
% Parameters:  S  - S-parameters of the 2-port network normalized to Z0
%              Z0 - (optional) reference impedance of the S-parameters;
%                   default 50 ohms
%              H  - hybrid parameters of the 2-port network
%              
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function H = S2H(S, Z0);

if (nargin < 1) | (nargin > 2)
    error('One or two input parameters are required!');
end
if ischar(S)
    error('First input parameter has to be numerical!');
end
[m,n] = size(S);
if n~=4
    error('Number of columns of the first input parameter has to be four!');
end

if nargin == 1
    Z0 = 50;	% Default-Systemimpedanz ist 50 Ohm
else
    if ischar(Z0)
        error('Second input parameter has to be numerical!');
    elseif length(Z0) ~= 1
        error('Second input parameter has to be scalar!');
    end
end

Nenner = (1-S(:,1)).*(1+S(:,4)) + S(:,2).*S(:,3);

H(:,1) = Z0 * ((1+S(:,1)).*(1+S(:,4)) - S(:,2).*S(:,3)) ./ Nenner;
H(:,2) =  2*S(:,2) ./ Nenner;
H(:,3) = -2*S(:,3) ./ Nenner;
H(:,4) = ((1-S(:,1)).*(1-S(:,4)) - S(:,2).*S(:,3)) ./ Nenner / Z0;
