%
% function NCircle = NoiseCircle(N, F, n);
%
% Calculates circles with constant noise figure, that can be
% drawn into a smith chart. Every column of the output parameter
% contains a circle. Noise parameters in "N" have to adhere to
% the common format, i.e. minimum noise figure (dB) in the first
% column, optimum source reflexion coefficient (normalized to
% 50 ohms) in the second column, normalized effective noise
% resistance (normalized to 50 ohms) in the third column.
%
% Parameters: N        -  noise parameters
%             F        -  noise figures of the circles in decibel
%             n        -  (optional) number of points per circle;
%                         default 51
%             NCircle  -  noise circles
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function NCircle = NoiseCircle(N, F, n);

if (nargin>3) | (nargin<2)
    error('Number of input parameters has to be 2 or 3!');
end
if ischar(N) | ischar(F)
    error('First and second parameter have to be numerical!');
end
[mN, nN] = size(N);
if nN ~= 3
    error('First parameter has to consist of 3 columns!');
end
if nargin == 2
    n = 51;
else
    if ischar(n)
        error('Third parameter has to be numerical!');
    elseif length(n) ~= 1
        error('Third parameter has to be scalar!');
    end
end

Fmin = 10.^(N(:,1)/10);
ropt = N(:,2);
Rn   = N(:,3)*50;

Gopt = real((1 - ropt) ./ (1 + ropt)) / 50;

F = 10.^(F/10);     % Rauschzahlen in linearen Mastab
K = exp(j*linspace(0,2*pi,n))';  % Einheitskreis mit n Punkten

for m=1:length(F)   % fr jede Rauschzahl berechnen
    NI = (F(m) - Fmin) .* (1-abs(ropt).^2) ./ (4*Rn.*Gopt);

    Radius = sqrt( NI.^2 + NI.*(1-abs(ropt).^2) ) ./ (1+NI);
    Mitte  = ropt ./ (1+NI);

    for k=1:mN  % fr jede Frequenz berechnen
        NCircle(:,k+(m-1)*mN) = Mitte(k) + Radius(k) * K;
    end
end
