%
% function [MAG, rG, rL] = MAGvalue(S);
%
% Calculates the maximum available gain (MAG) of a 2-port network with
% S-parameter matrix S. As this value is only defined for unconditional
% stable 2-port networks, infinity is obtained in all other cases.
% Optionally the reflexion coefficients of the generator and of the load,
% where the gain is achieved, can be received. Because the absolute values
% of these reflexion coefficients can be above one (instability), they
% should always be examined.
%
% Parameters:   S   -  S-parameters of a 2-port network
%               MAG -  MAG of 2-port network (linear, not in decibel)
%               rG  -  (optional) reflexion coefficient of the generator
%                      needed for MAG
%               rL  -  (optional) reflexion coefficient of the load needed
%                      for MAG
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function [MAG, rG, rL] = MAGvalue(S);

if nargin ~= 1
    error('One input parameter has to be delivered!');
end
if ischar(S)
    error('Input parameter has to be numerical!');
end
[m, n] = size(S);
if n ~= 4
    error('MAG is only defined for 2-port networks!');
end

MAG(1:m) = 0;
det      = S(:,1).*S(:,4) - S(:,2).*S(:,3);   % Determinant

m1       = (S(:,2) == 0);     % unilateral cases
MAG( m1) = abs(S(m1,3)).^2 ./ (1-abs(S(m1,1)).^2) ./ (1-abs(S(m1,4)).^2);

m1       = (~m1) & (S(:,3) ~= 0);
K(m1,1)  = (1 - abs(S(m1,1)).^2 - abs(S(m1,4)).^2 + abs(det(m1,1)).^2) ./ (2*abs(S(m1,2).*S(m1,3)));    % K-Faktor
MAG(m1)  = inf;
m2       = m1 & (K(:,1) >= 1);
MAG(m2)  = abs(S(m2,3) ./ S(m2,2)) .* (K(m2,1)-sqrt(K(m2,1).^2 - 1));

if nargout > 1
    rG(1:m) = 0;
    A       = 1 + abs(S(:,1)).^2 - abs(S(:,4)).^2 - abs(det).^2;
    B       = S(:,1) - conj(S(:,4)) .* det;
    rG(m2)  = (A(m2) - sqrt(A(m2).^2 - abs(2*B(m2)).^2)) ./ (2*B(m2));  % Generator-Reflexionsfaktor
    
    if nargout > 2
        rL(1:m) = 0;
        A       = 1 + abs(S(:,4)).^2 - abs(S(:,1)).^2 - abs(det).^2;
        B       = S(:,4) - conj(S(:,1)) .* det;
        rL(m2)  = (A(m2) - sqrt(A(m2).^2 - abs(2*B(m2)).^2)) ./ (2*B(m2));  % Last-Reflexionsfaktor
    end
end
