%
% function [K, d] = Kfactor(S);
%
% Calculates the K-factor (stability factor by Rollet) and the absolute
% value of the S-parameter determinant (auxiliary condition). A 2-port
% network is unconditionally stable, if and only if, the K-factor is above
% one and the auxiliary condition is less than one.
%
% Parameters:   S  -  S-parameter of the 2-port network
%               K  -  K-factor
%               d  -  auxiliary condition
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function [K, d] = Kfactor(S);

if nargin ~= 1
    error('One input parameter has to be delivered!');
end
if ischar(S)
    error('Input parameter has to be numerical!');
end
[m, n] = size(S);
if n ~= 4
    error('Stability factor is only defined for 2-port networks!');
end

m     = (S(:,2).*S(:,3) ~= 0);      % treat unilateral 2-ports separately
d     = abs(S(:,1).*S(:,4) - S(:,2).*S(:,3));   % absolute value of determinant
h     = 1 - abs(S(:,1)).^2 - abs(S(:,4)).^2 + d.^2;
K( m) = h(m,1) ./ (2*abs(S(m,2).*S(m,3)));    % K-factor

n     = ((~m) & (h == 0));  % special cases
K( n) = -1;
n     = ((~m) & (h ~= 0));
K( n) = sign(h(n)) * inf;
