%
% function S = H2S(H, Z0);
%
% Transforms hybrid parameters (H-parameters) of a 2-port network to
% S-parameters.
%
% Parameters:  H  - hybrid parameters of the 2-port network
%              Z0 - (optional) reference impedance of the S-parameters;
%                   default 50 ohms
%              S  - S-parameters of the 2-port network normalized to Z0
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function S = H2S(H, Z0);

if (nargin < 1) | (nargin > 2)
    error('One or two input parameters have to be delivered!');
end
if ischar(H)
    error('First parameter has to be numerical!');
end
[m,n] = size(H);
if n~=4
    error('Number of columns of the first parameter has to be four!');
end

if nargin == 1
    Z0 = 50;	% default reference impedance is 50 ohms
else
    if ischar(Z0)
        error('Second parameter has to be numerical!');
    elseif length(Z0) ~= 1
        error('Second parameter has to be scalar!');
    end
end

H(:,1) = H(:,1) / Z0;
H(:,4) = H(:,4) * Z0;
Nenner = (1+H(:,1)).*(1+H(:,4)) - H(:,2).*H(:,3);

S(:,1) = ((H(:,1)-1).*(H(:,4)+1) - H(:,2).*H(:,3)) ./ Nenner;
S(:,2) =  2*H(:,2) ./ Nenner;
S(:,3) = -2*H(:,3) ./ Nenner;
S(:,4) = ((1+H(:,1)).*(1-H(:,4)) + H(:,2).*H(:,3)) ./ Nenner;
