%
% function S = A2S(A, Z0);
%
% Transforms chain parameters (ABCD parameters) of a 2-port network into
% S-parameters.
%
% Parameters:  A  - chain parameter of the 2-port network
%              Z0 - (optional) reference impedance of the S-parameter;
%                   default 50 ohms
%              S  - S-parameter of the 2-port network normalized to Z0
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, developed with MATLAB 6.0.0.88 Release 12
%

function S = A2S(A, Z0);

if (nargin < 1) | (nargin > 2)
    error('One or two input parameters have to be delivered!');
end
if ischar(A)
    error('First parameter has to be numerical!');
end
[m,n] = size(A);
if n~=4
    error('Number of columns of the first parameterhas to be four!');
end

if nargin == 1
    Z0 = 50;	% default reference impedance is 50 ohms
else
    if ischar(Z0)
        error('Second parameter has to be numerical!');
    elseif length(Z0) ~= 1
        error('Second parameter has to be scalar!');
    end
end

Nenner = A(:,1) + A(:,4) + A(:,2)/Z0 + A(:,3)*Z0;

S(:,1) = (A(:,1) - A(:,4) + A(:,2)/Z0 - A(:,3)*Z0) ./ Nenner;
S(:,2) = 2*(A(:,1).*A(:,4) - A(:,2).*A(:,3)) ./ Nenner;
S(:,3) = 2 ./ Nenner;
S(:,4) = (A(:,4) - A(:,1) + A(:,2)/Z0 - A(:,3)*Z0) ./ Nenner;
