%
% function [Kreis, Region] = StabKreisG(S, n);
%
% Berechnet die Punkte von Stabilittskreisen des Generators. Optional
% wird die Region der Stabilitt zurckgegeben. ('i' = innerhalb des
% Kreises, 'a' = auerhalb des Kreises)
%
% Parameter:  S      -  S-Parameter eines 2-Tores
%             n      -  (optional) Anzahl der Punkte pro Kreis;
%                       default 51
%             Kreis  -  Punkte der Stabilittskreise in der Reflexions-
%                       faktor-Ebene des Generators
%             Region - Region der Stabilitt (innen/auerhalb)
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, entwickelt mit MATLAB 6.0.0.88 Release 12
%

function [Kreis, Region] = StabKreisG(S, nn);

if (nargin<1) | (nargin>2)
    error('Anzahl der Parameter mu 1 oder 2 betragen!');
end
if ischar(S)
    error('Erster Parameter mu numerisch sein!');
end
[m, n] = size(S);
if n ~= 4
    error('Erster Parameter mu 4 Spalten besitzen!');
end
if nargin == 1
    nn = 51;
else
    if ischar(nn)
        error('Zweiter Parameter mu numerisch sein!');
    elseif length(nn) ~= 1
        error('Zweiter Parameter mu skalar sein!');
    end
end

detS   = S(:,1).*S(:,4) - S(:,2).*S(:,3); % Determinante
Mitte  = (conj(detS).*S(:,4) - conj(S(:,1))) ./ (abs(detS).^2 - abs(S(:,1)).^2);
Radius = abs(S(:,2).*S(:,3) ./ (abs(detS).^2 - abs(S(:,1)).^2));

K = exp(j*linspace(0,2*pi,nn))';  % Einheitskreis mit n Punkten
for k=1:m  % fr jede Frequenz einen Kreis berechnen
    Kreis(:,k) = Mitte(k) + Radius(k) * K;
end

if nargout > 1  % Stabilittsregionen berechnen ?
    Region(1:m) = 'a';
    m1  = abs(Mitte)  < Radius;
    m2  = abs(S(:,4)) < 1;
    Region( m1 & m2) = 'i';
    Region(~(m1|m2)) = 'i';
end
