%
% function SpeicherSnP(DateiName, freq, S, freqN, N);
%
% Erzeugt eine Touchstone-Datei (SnP-Datei, z.B. FHC40.s2p). Die
% bergebene S-Parameter-Matrix mu pro Spalte einen S-Parameter
% enthalten, in der Reihenfolge S11, S12, ..., S21, S22, ... .
% Die S-Parameter mssen auf 50 Ohm normiert sein.
% Die Rauschparameter mssen (falls vorhanden) in jeder Zeile
% folgende Spalten aufweisen: minimale Rauschzahl in Dezibel,
% optimaler Quellenreflexionsfaktor normiert auf 50 Ohm,
% normalisierter effektiver Rauschwiderstand (50 Ohm normiert).
%
% Parameter:   DateiName  -  Name der zu erstellenden Datei
%              freq       -  Vektor mit Frequenz der S-Parameter
%              S          -  Matrix mit komplexen S-Parametern
%              freqN      -  optional: Frequenz der Rauschparameter
%              N          -  optional: Rauschparameter
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, entwickelt mit MATLAB 6.0.0.88 Release 12
%

function SpeicherSnP(DateiName, freq, S, freqN, N);

if (nargin~=3) & (nargin~=5)
    error('Es mssen drei oder fnf Parameter bergeben werden!');
end
if ~ischar(DateiName)
   error('Ester Parameter mu ein String sein!');
end
if ischar(freq) | ischar(S)
   error('Zweiter und dritter Parameter mssen numerisch sein!');
end

[m, n] = size(freq);
if (m~=1) & (n~=1)
   error('Zweiter Parameter mu 1-dimensional sein!');
end
[m, n] = size(S);
if m ~= length(freq)
    error('Zeilen-Anzahl des dritten Parameters mu gleich der Gre des zweiten sein!');
end
if n > 100
    error('Maximale Anzahl der S-Parameter-Tore ist 10!');
end
if mod(sqrt(n),1) ~= 0
    error('Spalten-Anzahl der S-Parameter mu 1,4,9,16,... (1,2,3,4-Tor etc.) sein!');
end
if nargin == 5
    if ischar(N)
        error('Vierter Parameter mu numerisch sein!');
    elseif n ~= 4
        error('Rausch-Parameter sind nur bei 2-Tor-S-Parameter sinnvoll!');
    end

    [u, v] = size(N);
    if v ~= 3
        error('Spalten-Anzahl der Rausch-Parameter mu 3 sein!');
    end
    if freq(length(freq)) < freqN
        error('Erste Rausch-Parameter-Frequenz mu kleiner/gleich der letzten S-Parameter-Frequenz sein!');
    end
    if length(freqN) ~= u
        error('Zeilen-Anzahl des fnften Parameters mu gleich der Gre des vierten sein!');
    end
end

h = fopen(DateiName, 'w');      % Datei erstellen
fprintf(h, '! ------------------------------------------------------------------------\r\n');
fprintf(h, '! Erstellt mit MATLAB-Funktion von M. Margraf\r\n');
fprintf(h, '! Created  by  MATLAB function by  M. Margraf\r\n');
fprintf(h, '! ------------------------------------------------------------------------\r\n');
fprintf(h, '! symbol freq-unit parameter-type data-format keyword impedance-ohm\r\n');
fprintf(h, '#        HZ        S              MA          R       50\r\n');

if n == 1       % 1-Tor ?
    fprintf(h, '! freq    magS11   angS11\r\n');
elseif n == 4   % 2-Tor ?
    S = [S(:,1) S(:,3) S(:,2) S(:,4)];  % bei 2-Tor-Parametern S12 und S21 vertauschen
    fprintf(h, '! freq    magS11     angS11      magS21      angS21      magS12      angS12      magS22       angS22\r\n');
else
    fprintf(h, '! freq    magnitude    angle  ....\r\n');
end

for i=1:length(freq)            % jede Daten-Zeile einzeln schreiben
    fprintf(h, '%g  ', freq(i));
    for j=1:n
        fprintf(h, '%g %g ', abs(S(i,j)), angle(S(i,j))*180/pi);
        if n<=4
            if j == n           % bei kleinerer Tor-Anzahl alle S-Parameter in einer Zeile
                fprintf(h, '\r\n');   % Zeilen-Umbruch
            end
        elseif mod(j, sqrt(n)) == 0   % bei grerer Tor-Anzahl in mehrere Zeilen verteilen
            fprintf(h, '\r\n');       % Zeilen-Umbruch
        end
    end
end

if nargin == 5
fprintf(h, '! Noise data:  freq    Fmin     Gamma-opt     rn\r\n');
for i=1:u       % falls vorhanden, Rausch-Parameter schreiben
    % optimaler Quellenreflexionsfaktor wird immer linear angegeben
    fprintf(h, '%g  %g %g %g %g\r\n', freqN(i), N(i,1), abs(N(i,2)), angle(N(i,2))*180/pi, N(i,3));
end
end

fclose(h);      % Datei schlieen
