%
% function SpeicherCITI(DateiName, unabh, abh, Namen);
%
% Erzeugt eine CITI-Datei (Common Instrumentation Transfer and
% Interchange). Die unabhngige Variable mu ein-dimensional sein.
% Die abhngige Variable darf zwei-dimensional sein. Wird bei den
% Daten direkt eine Variable bergeben, so erhalten die Daten in
% der CITI-Datei den Namen dieser Variablen; ansonsten werden die
% Namen "unabhVar" und "abhVar" verwendet. Fr die abhngige
% Variable kann der Namen in einer Zeichenkette bergeben werde.
% Das ist insbesondere bei einer zwei-dimensionalen abhngigen
% Variablen sinnvoll.
% Beispiel: SpeicherCITI('daten.cit', Ugs, Dat, ['n1';'n2';'nr3'])
%
% Parameter:   DateiName   - Name der zu erstellenden Datei
%              unabh       - unabhngige Variable
%              abh         - abhngige Variable(n)
%              Namen       - (optional) Variablennamen fr die
%                            abhngige(n) Variable(n)
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, entwickelt mit MATLAB 6.0.0.88 Release 12
%

function SpeicherCITI(DateiName, unabh, abh, Namen);

if (nargin<3) | (nargin>4)
    error('Es mssen 3 oder 4 Parameter bergeben werden!');
end
if ~ischar(DateiName)
    error('Erster Parameter mu ein String sein!');
end
if ischar(unabh) | ischar(abh)
    error('Zweiter und dritter Parameter mu numerisch sein!');
end

[mi, ni] = size(unabh);
if ni > 1
    if mi == 1
        unabh = conj(unabh');    % in Zeilenvektor wandeln
        abh = conj(abh');        % in Zeilenvektor wandeln (der Einheit wegen)
    else
        error('Unabhngige Variable darf nur ein-dimensional sein!');
    end
end

[md, nd] = size(abh);
if nd > 1
    if md == 1
        abh = conj(abh');     % in Zeilenvektor wandeln
    end
end

[md, nd] = size(abh);
if md ~= length(unabh)
    error('Lnge der Daten-Vektoren mu gleich sein!');
end

if nargin==4
    if ~ischar(Namen)
        error('Vierter Parameter mu ein String sein!');
    end
    [ms, ns] = size(Namen);
    if nd ~= ms
        error('Falsche Anzahl von Namen!');
    end
else
    if isempty(inputname(3))
        s = 'abhVar';      % wenn ntig, Namen selber vergeben
    else
        s = inputname(3);
    end
    if nd>1
        for m=1:nd
            Namen(m,:) = sprintf('%s%i',s,m);
        end
    else
        Namen = s;
    end
end

% ************************************************************************************
fid = fopen(DateiName, 'w');      % Datei erstellen
fprintf(fid, 'CITIFILE A.01.00\r\n');   % Kopfzeile schreiben
fprintf(fid, 'NAME DATA\r\n');

if ~isreal(unabh)
    Typen = 'RI';       % Variable ist complex
else
    Typen = 'MAG';      % Variable ist reell
end

if isempty(inputname(2))    % wenn keine Variable bergeben wurde ...
    Name = 'unabhVar';      % ... selber einen Namen vergeben
else
    Name = inputname(2);
end

fprintf(fid, 'VAR  %s %s %i\r\n', Name, Typen, length(unabh));

if ~isreal(abh)
    Typen = 'RI';       % Variable ist complex
else
    Typen = 'MAG';      % Variable ist reell
end
    
for m=1:nd
    fprintf(fid, 'DATA %s %s %i\r\n', Namen(m,:), Typen, md);
end

% unabhngige Variable in Datei schreiben
fprintf(fid, 'VAR_LIST_BEGIN\r\n');
if isreal(unabh)
    fprintf(fid, '%g\r\n',unabh);
else
    for n=1:length(unabh)
        fprintf(fid, '%g, %g\r\n',real(unabh(n)), imag(unabh(n)));
    end
end
fprintf(fid, 'VAR_LIST_END\r\n');

% abhngige Variable in Datei schreiben
for n=1:nd
    fprintf(fid, 'BEGIN\r\n');
    if isreal(abh)
        fprintf(fid, '%g\r\n',abh(:,n));
    else
        for m=1:md
            fprintf(fid, '%g, %g\r\n',real(abh(m,n)), imag(abh(m,n)));
        end
    end
    fprintf(fid, 'END\r\n');
end

fclose(fid);
