%
% function S_Z1 = Snorm(S_Z0, Z0, Z1);
%
% Rechnet die S-Parameter S_Z0 mit der Referenz-Impedanz Z0 in
% S-Parameter S_Z1 mit der Referenz-Impedanz Z1 um. Falls Z0
% und/oder Z1 nicht fr alle Ports gleich ist, mu ein Vektor
% bergeben werden, der fr jeden Port eine Impedanz enthlt.
% Wird eine skalare Impedanz bergeben, so gilt sie fr alle
% Ports.
%
% Parameter:   S_Z0 -  S-Parameter auf Z0 normiert
%              Z0   -  Referenz-Impedanz der Eingabe-S-Parameter
%                      (Vektor oder skalar)
%              Z1   -  Referenz-Impedanz der Ausgabe-S-Parameter
%                      (Vektor oder skalar)
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, entwickelt mit MATLAB 6.0.0.88 Release 12
%

function S_Z1 = Snorm(S_Z0, Z0, Z1);

if nargin ~= 3
    error('Es mssen drei Parameter bergeben werden!');
end
if ischar(S_Z0) | ischar(Z0) | ischar(Z1)
    error('Eingabe-Parameter mssen numerisch sein!');
end
[m, n] = size(S_Z0);
q      = sqrt(n);
if mod(q,1) ~= 0
    error('Spalten-Anzahl des erster Parameters mu quadratisch sein!');
end
if (length(Z0)~=1) & (length(Z0)~=q)
    error('Ungltige Gre des zweiten Parameters!');
end
if (length(Z1)~=1) & (length(Z1)~=q)
    error('Ungltige Gre des dritten Parameters!');
end
if any(Z0<=0) | any(Z1<=0)
    error('Normierungen auf kleiner oder gleich 0 Ohm ergeben keinen Sinn!');
end

L0 = length(Z0);
if L0==1            % falls fr alle Ports gleich ...
    Z0(1:q) = Z0;   % ..., dann trotzdem Vektor erzeugen
end
L1 = length(Z1);
if L1 ==1           % falls fr alle Ports gleich ...
    Z1(1:q) = Z1;   % ..., dann trotzdem Vektor erzeugen
end
r = zeros(q);
for w=1:q       % Diagonal-Matrix erzeugen
    r(w,w) = (Z1(w)-Z0(w)) / (Z1(w)+Z0(w));  % formaler Reflexionsfaktor
end

for v=1:m
    for w=0:n-1   % quadratische S-Parameter-Matrix erstellen
        S(floor(w/q)+1, mod(w,q)+1) = S_Z0(v,w+1);
    end

    S = (S-r) * inv((eye(q)-r*S));   % S-Parameter umnormieren

    for w=0:q-1   % in bliche Matrixform (nicht quadratisch) schreiben
        S_Z1(v,w*q+1:w*q+q) = S(w+1,:);
    end
end

if (L0==1) & (L1==1) return; end    % alle Port-Impedanzen identisch, dann folgende Normung unntig
for w=0:n-1     % an jedem Port auf neue Impedanz normieren
    zeile  = floor(w/q)+1;
    spalte = mod(w,q)+1;
    S_Z1(:,w+1) = S_Z1(:,w+1) * sqrt(Z1(zeile)/Z0(zeile) * Z0(spalte)/Z1(spalte)) * (Z1(zeile)+Z0(zeile))/Z1(zeile) * Z1(spalte)/(Z1(spalte)+Z0(spalte));
end
