%
% function Kreis = RauschKreise(N, F, n);
%
% Berechnet die Punkte von Kreisen konstanter Rauschzahl, die
% dann in einem Smith-Diagramm dargestellt werden knnen, d.h.
% jede Spalte des Rckgabeparameters enthlt einen Kreis.
% Die Rauschparameter in N mssen das blich Format aufweisen,
% d.h. minimale Rauschzahl (dB) in der ersten Spalte, optimaler
% Quellenreflexionsfaktor normiert auf 50 Ohm in der zweiten
% Spalte und normalisierter effektiver Rauschwiderstand  (50
% Ohm normiert)in der dritten Spalte.
%
% Parameter:  N     -  Rauschparameter
%             F     -  Rauschzahlen der Kreise in Dezibel
%             n     -  (optional) Anzahl der Punkte pro Kreis;
%                      default 51
%             Kreis -  Punkte der Rauschkreise
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, entwickelt mit MATLAB 6.0.0.88 Release 12
%

function Kreis = RauschKreise(N, F, n);

if (nargin>3) | (nargin<2)
    error('Anzahl der Parameter mu 2 oder 3 betragen!');
end
if ischar(N) | ischar(F)
    error('Erster und zweiter Parameter mu numerisch sein!');
end
[mN, nN] = size(N);
if nN ~= 3
    error('Erster Parameter mu 3 Spalten besitzen!');
end
if nargin == 2
    n = 51;
else
    if ischar(n)
        error('Dritter Parameter mu numerisch sein!');
    elseif length(n) ~= 1
        error('Dritter Parameter mu skalar sein!');
    end
end

Fmin = 10.^(N(:,1)/10);
ropt = N(:,2);
Rn   = N(:,3)*50;

Gopt = real((1 - ropt) ./ (1 + ropt)) / 50;

F = 10.^(F/10);     % Rauschzahlen in linearen Mastab
K = exp(j*linspace(0,2*pi,n))';  % Einheitskreis mit n Punkten

for m=1:length(F)   % fr jede Rauschzahl berechnen
    NI = (F(m) - Fmin) .* (1-abs(ropt).^2) ./ (4*Rn.*Gopt);

    Radius = sqrt( NI.^2 + NI.*(1-abs(ropt).^2) ) ./ (1+NI);
    Mitte  = ropt ./ (1+NI);

    for k=1:mN  % fr jede Frequenz berechnen
        Kreis(:,k+(m-1)*mN) = Mitte(k) + Radius(k) * K;
    end
end
