%
% function [MAG, rG, rL] = MAGwert(S);
%
% Berechnet die maximal verfgbare Leistungsverstrkung (MAG) fr ein
% 2-Tor mit der S-Parameter-Matrix S. Da dieser Wert nur fr absolut
% stabile 2-Tore definiert ist, wird in allen anderen Fllen unendlich
% zurckgeliefert. Optional knnen die Generator- und Last-Reflexions-
% faktoren ermittelt werden, bei denen die angegebene Verstrkung
% erreicht wird. Da die Betrge der Reflexionsfaktoren auch grer eins
% sein knnen (Instabilitt), sollten sie stets berprft werden!
%
% Parameter:    S   -  2-Tor-S-Parameter
%               MAG -  MAG des 2-Tores (linear, nicht in Dezibel)
%               rG  -  (optional) Generator-Reflexionsfaktor fr MAG
%               rL  -  (optional) Last-Reflexionsfaktor fr MAG
%
% Microwave Toolbox 2.6, M. Margraf, margraf@mwt.ee.tu-berlin.de
% Freeware, entwickelt mit MATLAB 6.0.0.88 Release 12
%

function [MAG, rG, rL] = MAGwert(S);

if nargin ~= 1
    error('Es mu genau ein Parameter bergeben werden!');
end
if ischar(S)
    error('Eingabe-Parameter mu numerisch sein!');
end
[m, n] = size(S);
if n ~= 4
    error('Maximal verfgbarer Gewinn ist nur fr 2-Tore definiert!');
end

MAG(1:m) = 0;
det      = S(:,1).*S(:,4) - S(:,2).*S(:,3);   % Determinante

m1       = (S(:,2) == 0);     % unilaterale Flle merken
MAG( m1) = abs(S(m1,3)).^2 ./ (1-abs(S(m1,1)).^2) ./ (1-abs(S(m1,4)).^2);

m1       = (~m1) & (S(:,3) ~= 0);
K(m1,1)  = (1 - abs(S(m1,1)).^2 - abs(S(m1,4)).^2 + abs(det(m1,1)).^2) ./ (2*abs(S(m1,2).*S(m1,3)));    % K-Faktor
MAG(m1)  = inf;
m2       = m1 & (K(:,1) >= 1);
MAG(m2)  = abs(S(m2,3) ./ S(m2,2)) .* (K(m2,1)-sqrt(K(m2,1).^2 - 1));

if nargout > 1
    rG(1:m) = 0;
    A       = 1 + abs(S(:,1)).^2 - abs(S(:,4)).^2 - abs(det).^2;
    B       = S(:,1) - conj(S(:,4)) .* det;
    rG(m2)  = (A(m2) - sqrt(A(m2).^2 - abs(2*B(m2)).^2)) ./ (2*B(m2));  % Generator-Reflexionsfaktor
    
    if nargout > 2
        rL(1:m) = 0;
        A       = 1 + abs(S(:,4)).^2 - abs(S(:,1)).^2 - abs(det).^2;
        B       = S(:,4) - conj(S(:,1)) .* det;
        rL(m2)  = (A(m2) - sqrt(A(m2).^2 - abs(2*B(m2)).^2)) ./ (2*B(m2));  % Last-Reflexionsfaktor
    end
end
